program Park;

(*
This solution to the parking lot problem was written by

     Dr. John M. Gwynn, Jr.
     Professor of Computer Science
     California State University, Sacramento

Input is from the text file InFileName, which is declared in the const 
section.  Output is to the screen.

The program contains code that determines if a vacated position is read after 
all waiting cars have been parked.  This halts the program and a message is 
displayed.  (This condition should not occur for correct data.)

CarArray contains the original positions of waiting cars in the order that 
the positions were input.  Subscript 1 contains the initial position of the 
first car input, subscript 2 the initial position of the second car 
input,..., subscript NCars the initial position of the last car input.

PosArray maintain the positions of waiting cars by car number.  0 indicates 
that no car is waiting at this position; a non-0 value N indicates that car 
number N is waiting at this position.

ParkArray maintains the positions in which the waiting cars park; the order 
of the cars is the same as in CarArray.  0 indicates that this car has not 
yet parked; a non-0 number N indicates that this car has parked in space N.

MoveArray is used to facilitate the updating of PosArray to reflect that the 
waiting cars move to fill a vacated position.

*)

  label
    EndProg;

  const
    InFileName = 'PARK.DAT';
    Sentinel = 99;
    MaxPos = 20;

  type
    ArrType = array[1..MaxPos] of integer;

  var
    PosArray, CarArray, MoveArray, ParkArray : ArrType;
    ICar, NCars, CarPos, ParkCar, Vacate, NMove : integer;
    FoundCar, NoCar : Boolean;
    FileIn : text;

begin
assign (FileIn, InFileName);
reset (FileIn);

(* Initialize PosArray, CarArray, and ParkArray to indicate that no cars 
   are waiting. *)
for ICar := 1 to MaxPos do
  PosArray[ICar] := 0;
for ICar := 1 to MaxPos do
  CarArray[ICar] := 0;
for ICar := 1 to MaxPos do
  ParkArray[ICar] := 0;

(* Read in the initial positions of the waiting cars and count the number of 
   cars read in NCars.  The sentinel ends the list of waiting cars. *)
NCars := 0;
readln (FileIn, CarPos);
while (CarPos <> Sentinel) do
  begin
  NCars := NCars + 1;
  PosArray[CarPos] := NCars;
  CarArray[NCars] := CarPos;
  readln (FileIn, CarPos);
  end; (* while CarPos <> Sentinel *)

(* Read in vacated spaces until end of file, filling each vacated space 
   before reading the next vacated space. *)
NoCar := false;
while (not (EOF (Filein))) and (not NoCar) do
  begin
  readln (FileIn, Vacate);
  (* Find the car nearest the vacated space in the direction of traffic 
     flow.  The search begins at the vacated position backward to position 1, 
     then continues from the maximum position backward to the position 
     immediately after vacated position.  When the nearest car is found, the 
     program computes the number of positions NMove that this car must be 
     moved forward to fill the vacated space.  NMove may be 0. *)
  FoundCar := false;
  ICar := Vacate;
  while (ICar >= 1) and (not FoundCar) do
    begin
    if (PosArray[ICar] <> 0) then
      begin
      ParkCar := PosArray[ICar];
      NMove := Vacate - ICar;
      FoundCar := true;
      end (* then *)
    else
      ICar := Icar - 1;
    end; (* while *)
  if (not FoundCar) then
    begin
    ICar := MaxPos;
    while (ICar >= Vacate + 1) and (not FoundCar) do
      begin
      if (PosArray[ICar] <> 0) then
        begin
        ParkCar := PosArray[ICar];
        NMove := MaxPos - ICar + Vacate;
        FoundCar := true;
        end (* then *)
      else
        ICar := Icar - 1;
      end; (* while *)
    end; (* then *)
  if (not FoundCar) then
    begin
    (* A position has been vacated after all cars have been parked. *)
    NoCar := true;
    end (* then *)
  else
    begin
    (* Update ParkArray to reflect that the nearest car parked in the vacated 
       position and PosArray to reflect that the nearest car is no longer 
       waiting. *)
    ParkArray[PosArray[ICar]] := Vacate;
    PosArray[ICar] := 0;
    (* Move all remaining cars forward NMove positions, wrapping around as 
       necessary.  The required shift in subscript positions is accomplished 
       by first copying subscript positions of PosArray to shifted subscript 
       positions of MoveArray and then copying MoveArray back to PosArray 
       without any shift. *)
    for ICar := 1 to MaxPos do
      begin
      if (ICar + NMove <= MaxPos) then
        MoveArray[ICar + NMove] := PosArray[ICar]
      else
        MoveArray[ICar + NMove - MaxPos] := PosArray[ICar];
      end; (* for *)
    for ICar := 1 to MaxPos do
      PosArray[ICar] := MoveArray[ICar];
    end; (* else *)
  end; (* while not EOF and not NoCar *)
  (* Since all vacated positions have been filled, display the output. *)
  for ICar := 1 to NCars do
    begin
    if (ParkArray[ICar] <> 0) then
      writeln ('Original position ', CarArray[ICar], ' parked in ', 
               ParkArray[ICar])
    else
      writeln ('Original position ', CarArray[ICar], ' did not park');
    end; (* for ICar *)
if NoCar then
  writeln ('no car found to fill vacant space ', Vacate);
end. (* program Park *)

